﻿/*==============================================================*/
/* DBMS name:      SQLite for Android                           */
/* Created on:     2013-07-12 15:00:00                          */
/*                                                              */
/* Podane typy kolumn, SQLite automatycznie konwertuje na swoje */
/* INTEGER, REAL, TEXT, BLOB zgodnie z opisem na stronie:       */
/* http://www.sqlite.org/datatype3.html                         */
/*                                                              */
/* "SQLite doesn't support the ADD CONSTRAINT variant of the    */
/* ALTER TABLE command (sqlite.org: SQL Features That SQLite    */
/* Does Not Implement)."                                        */
/* Z tego powodu dodawanie dodawanie foreign keys na razie sobie*/
/* odpuściłem, bo musiałbym posortować tablice. Poza tym        */
/* Foreign keys są obsługiwane dopiero od SQLite wer. 3.6.19.   */
/*                                                              */
/*==============================================================*/

/*======================================================================================================*/
/* ZMIANY STRUKTURY:                                                                                    */
/*|Data      |StructureVersion|Opis                                                                     */
/* ---------------------------------------------------------------------------------------------------- */
/* 2014-10-24|             42 |dodano Product.Favorite                                                  */
/* 2014-11-25|             43 |dodano ProductGroup.HiddenOnPanels                                       */
/*       -//-|           -//- |dodano ProductGroup.HiddenOnBiller                                       */
/* 2015-02-26|             44 |dodano Product.ImageLink                                                 */
/* 2015-08-04|             45 |dodano Product.Precission                                                */
/*       -//-|           -//- |dodano Product.SoldAmount                                                */
/*       -//-|           -//- |dodano Product.SoldValue                                                 */
/* 2015-08-27|             46 |dodano Barcode.Price                                                     */
/* 2015-10-01|             47 |dodano tablicę VatRateLtr                                                */
/* 2015-10-02|             48 |dodano ProductId, BarcodeId                                              */
/*       -//-|           -//- |dodano ReceiptItemReceiptProductId                                       */
/* 2015-10-21|             49 |dodano Product.StockAmount                                               */
/*       -//-|           -//- |dodano Product.StockBlocked                                              */
/* 2016-02-24|             50 |dodano FiscalReportVat.Type                                              */
/* 2016-09-05|             51 |dodano Receipt.CustomerId2Type                                           */
/*       -//-|           -//- |dodano Receipt.CustomerId2                                               */
/* 2016-09-13|             52 |dodano ProductGroup.IsExisable                                           */
/* 2017-01-13|             53 |dodano GastroElementList.IsPriceFromProduct                              */
/*       -//-|           -//- |dodano GastroElement.IsDifferentPrice                                    */
/* 2017-02-07|             54 |dodano Receipt.PrintoutNumber                                            */
/* 2017-03-07|             55 |dodano Barcode.CodeTango                                                 */
/* 2017-04-18|             56 |dodano Product.KitchenPrinterNumbers                                     */
/* 2017-05-09|             57 |dodano tablicę FiscalReportECopy                                         */
/* 2017-05-17|             58 |dodano Receipt.TaxPayerId                                                */
/* 2017-05-19|             59 |brak zmian tutaj                                                         */
/* 2017-08-07|             60 |dodano tablicę GastroTransition                                          */
/* 2017-08-29|             61 |dodano tablicę CustomerCardFormat                                        */
/*       -//-|           -//- |dodano DiscountLevel.Name                                                */
/* 2017-10-02|             62 |dodano tablicę DiscountDefined                                           */
/* 2017-10-10|             63 |dodano pola JPK w tablicy Customer                                       */
/* 2017-10-18|             64 |brak zmian tutaj                                                         */
/* 2017-10-23|             65 |dodano CustomerCardFormat.PrefixPosition                                 */
/*       -//-|           -//- |dodano CustomerCardFormat.CustomerCodePosition                           */
/* 2017-11-30|             66 |dodano CustomerCardFormat.Name                                           */
/* 2018-01-10|             67 |nowa tablica RadDiscountDefinition                                       */
/* 2018-11-26|             68 |nowe pole w tablicy Receipt - VatMarkup                                  */
/* 2018-12-27|             69 |nowe pole ReceiptItem.VatRateVMOrg                                       */
/* 2019-05-01|             70 |nowe pola w ReceiptItemRule                                              */
/*       -//-|           -//- |     ReceiptItemRule.ReceiptPointsOrdinal                                */
/*       -//-|           -//- |     ReceiptItemRule.Number2                                             */
/*       -//-|           -//- |     ReceiptItemRule.Number3                                             */
/*       -//-|           -//- |     ReceiptItemRule.Number4                                             */
/*       -//-|           -//- |     ReceiptItemRule.Int1                                                */
/*       -//-|           -//- |     ReceiptItemRule.Int2                                                */
/*       -//-|           -//- |nowe pola w ReceiptPoints:                                               */
/*       -//-|           -//- |     ReceiptPoints.Number_2                                              */
/*       -//-|           -//- |     ReceiptPoints.Number_3                                              */
/*       -//-|           -//- |     ReceiptPoints.Int_2                                                 */
/*       -//-|           -//- |nowa tabela PointsRuleProductCode                                        */
/* 2019-05-16|           -//- |nowa tabela Image                                                        */
/* 2019-05-16|           -//- |nowa tabela ImageParam                                                   */
/* 2019-05-16|           -//- |nowa tabela ImageRelation                                                */
/* 2019-07-07|             71 |nowa tabela ReceiptPaymentBon                                            */
/* 2019-09-26|             72 |nowe pole w GastroElement.GastroElementOrdinal                           */
/* 2019-11-15|             73 |nowe pole Product.IsSplitPayment                                         */
/* 2019-12-02|             74 |nowe pole Customer.VatPayerStatus                                        */
/*       -//-|           -//- |nowe pole Customer.VatPayerStatusDate                                    */
/* 2019-12-16|             75 |nowa tabela ReceiptPointsParam                                           */
/* 2020-01-30|             76 |nowa tabela Phrase                                                       */
/* 2020-10-06|             77 |nowe pole Product.GTU                                                    */
/* 2020-10-23|             78 |nowe pole ReceiptItem.DiscountSource                                     */
/* 2020-11-12|             79 |nowe pole Phrase.PriceLevel                                              */
/* 2021-09-15|             80 |zmiana długości Phrase.Text z 60 do 255                                  */
/* 2021-11-22|             81 |dodanie tabel dla monitoringu                                            */
/* 2022-02-11|             82 |dodanie tabel na definicje kodów                                         */
/* 2024-06-24|             83 |nowe pola Receipt.NielsenExported, Receipt.NielsenStopExport             */
/* 2024-10-07|             84 |nowe tabele ProdOpak,FrakcjaDRS,PosOpakDRS,PosDaneOpakDRS,PosKuponDRS    */
/* 2025-03-14|             85 |nowe pole Product.FrakId                                                 */
/*======================================================================================================*/

/*==============================================================*/
/* Table: AddReceiptOperator                                    */
/*==============================================================*/
create table AddReceiptOperator (
   PosId                numeric(9)           not null,
   ReceiptId            numeric(9)           not null,
   OperatorId           numeric(9)           not null,
   constraint PK_ADDRECEIPTOPERATOR primary key (OperatorId, ReceiptId, PosId)
)
go

/*==============================================================*/
/* Table: Barcode - kody towarow                                */
/*==============================================================*/
create table Barcode (
   Code                 text(40)             not null,
   ProductId            numeric(9)           not null,
   IsDefault            smallint             not null,
   Multiplier           decimal(10,4)        not null,
   ProductName          varchar(120)         null,
   PriceLevel           smallint             null,
   Price                decimal(14,2)        null,
   CodeTango            text(40)             null,
   constraint PK_BARCODE primary key (Code, ProductId)
)
go

/*==============================================================*/
/* Index: BarcodeId                                             */
/*==============================================================*/
create index BarcodeId on Barcode (
ProductId ASC,
IsDefault ASC
)
go

/*==============================================================*/
/* Table: BarcodeFormat - format kodu wazonego                  */
/*==============================================================*/
create table BarcodeFormat (
   BarcodeFormatId      INTEGER              PRIMARY KEY AUTOINCREMENT,
   IsActive             smallint             not null,
   Name                 text(40)             not null,
   Prefix               text(10)             not null,
   NumberLength         int                  not null,
   PluGroupIdentifier   int                  null,
   PriceVerifier        smallint             null,
   WeighedInterpretation int                  null,
   WeighedPrecision     int                  null,
   ExternalId           text(20)             null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null
)
go

/*==============================================================*/
/* Table: CCCurrency                                            */
/*==============================================================*/
create table CCCurrency (
   CCId                 numeric(9)           not null,
   OperatorId           numeric(9)           not null,
   ShiftId              numeric(9)           not null,
   CurrencyId           numeric(9)           not null,
   CurrencyRate         decimal(14,4)        null,
   constraint PK_CCCURRENCY primary key (CCId, OperatorId, ShiftId, CurrencyId)
)
go

/*==============================================================*/
/* Table: CCPayment                                             */
/*==============================================================*/
create table CCPayment (
   PaymentFormId        numeric(9)           not null,
   CCId                 numeric(9)           not null,
   OperatorId           numeric(9)           not null,
   ShiftId              numeric(9)           not null,
   CurrencyId           numeric(9)           not null,
   Total                decimal(14,2)        null,
   TotalCashier         decimal(14,2)        null,
   PaymentCount         int                  null,
   PaymentIn            decimal(14,2)        null,
   PaymentOut           decimal(14,2)        null,
   Withdrawal           decimal(14,2)        null,
   PaymentOutToBank     decimal(14,2)        null,
   PaymentOutOnRefund   decimal(14,2)        null,
   PaymentInCount       int                  null,
   PaymentOutCount      int                  null,
   PaymentOutToBankCount int                 null,
   PaymentOutOnRefundCount int               null,
   PaymentInOnOpen      decimal(14,2)        null,
   PaymentOutOnClose    decimal(14,2)        null,
   constraint PK_CCPAYMENT primary key (PaymentFormId, CCId, OperatorId, ShiftId, CurrencyId)
)
go

/*==============================================================*/
/* Table: CCRateLog                                             */
/*==============================================================*/
create table CCRateLog (
   CCId                 numeric(9)                     not null,
   CashierId            numeric(9)                     not null,
   ShiftId              numeric(9)                     not null,
   CurrencyId           numeric(9)                     not null,
   Type                 int                            not null,
   Rate                 decimal(15,4)                  null,
   Date                 datetime                       null,
   TableNumber          varchar(60)                    null,
   constraint PK_CCRATELOG primary key (CCId, CashierId, ShiftId, CurrencyId, Type)
)
go

/*==============================================================*/
/* Table: CashierClearing - rozliczone zmiany kasjerow          */
/*==============================================================*/
create table CashierClearing (
   CCId                 INTEGER              PRIMARY KEY AUTOINCREMENT,
   CashierId            numeric(9)           not null,
   ShiftId              numeric(9)           not null,
   OperatorId           numeric(9)           null,
   OperationTime        datetime             null,
   IsActive             smallint             null,
   Type                 int                  null,
   ShiftStart           datetime             null,
   ShiftStop            datetime             null,
   TotalCashIn          decimal(14,2)        null,
   TotalCashOut         decimal(14,2)        null,
   TotalDiscount        decimal(14,2)        null,
   TotalReceiptCount    int                  null,
   TotalCanceledReceiptCount int                  null,
   TotalCanceledReceiptValue decimal(14,2)        null,
   TotalStornoCount     int                  null,
   TotalStornoValue     decimal(14,2)        null,
   TotalDeposit         decimal(14,2)        null,
   TotalDepositReturned decimal(14,2)        null,
   TotalDrawerOpenCount int                  null,
   TotalBreakInSec      decimal(9)           null,
   TotalReceiptCouponCount int                  null,
   TotalReceiptCardCount int                  null,
   TotalReceiptCashCount int                  null,
   TotalBillCount       int                  null,
   TotalBillValue       decimal(14,2)        null,
   TotalPrepaidCount    int                  null,
   TotalPrepaidValue    decimal(14,2)        null,
   TotalPercentDiscount decimal(14,2)        null,
   TotalValueDiscount   decimal(14,2)        null,
   ExternalId           text(20)             null,
   AutoShiftClearingType smallint            null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null
)
go

/*==============================================================*/
/* Index: CCExtIdIndex                                          */
/*==============================================================*/
create index CCExtIdIndex on CashierClearing (
ExternalId ASC
)
go

/*==============================================================*/
/* Table: Category                                              */
/*==============================================================*/
create table Category (
   CategoryId           INTEGER              PRIMARY KEY AUTOINCREMENT,
   Name                 text(80)             null,
   ExternalId           text(20)             null,
   IsActive             smallint             null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null
)
go

/*==============================================================*/
/* Index: CategoryExtId                                         */
/*==============================================================*/
create index CategoryExtId on Category (
ExternalId ASC
)
go

/*==============================================================*/
/* Table: CategoryValue                                         */
/*==============================================================*/
create table CategoryValue (
   CategoryValueId      INTEGER              PRIMARY KEY AUTOINCREMENT,
   CategoryId           numeric(9)           null,
   Name                 text(80)             null,
   ExternalId           text(20)             null
)
go

/*==============================================================*/
/* Index: CategoryValueExtId                                    */
/*==============================================================*/
create index CategoryValueExtId on CategoryValue (
ExternalId ASC
)
go

/*==============================================================*/
/* Table: Config                                                */
/*==============================================================*/
create table Config (
   ParamGroup           text(255)            not null,
   ParamName            text(255)            not null,
   Ordinal              int                  not null,
   ParamValue           text(255)            not null,
   LastUpdate           datetime             not null,
   constraint PK_CONFIG primary key (ParamGroup, ParamName, Ordinal)
)
go

/*==============================================================*/
/* Table: Contract                                              */
/*==============================================================*/
create table Contract (
   ContractId           INTEGER              PRIMARY KEY AUTOINCREMENT,
   CustomerId           numeric(9)           null,
   Name                 text(120)            null,
   IsActive             smallint             null,
   ContractType         smallint             null,
   ContractStart        datetime             null,
   ContractEnd          datetime             null,
   Priority             smallint             null,
   Desc1                varchar(60)          null,
   Desc2                varchar(60)          null,
   Desc3                varchar(60)          null,
   Desc4                varchar(60)          null,
   ExternalId           text(20)             null,
   Creation             datetime             null,
   LastUpdate           datetime             null
)
go

/*==============================================================*/
/* Index: ContractExtIdIndex                                    */
/*==============================================================*/
create index ContractExtIdIndex on Contract (
ExternalId ASC
)
go

/*==============================================================*/
/* Index: ContractContractStartContractEndInx                   */
/*==============================================================*/
create index ContractContractStartContractEndInx on Contract ( ContractStart, ContractEnd)
go

/*==============================================================*/
/* Table: ContractProducts                                      */
/*==============================================================*/
create table ContractProducts (
   ContractId           numeric(9)           not null,
   ProductId            numeric(9)           not null,
   Price                decimal(14,2)        null,
   PercentDiscount      decimal(14,4)        null,
   Creation             datetime             null,
   LastUpdate           datetime             null,
   constraint PK_CONTRACTPRODUCTS primary key (ProductId, ContractId)
)
go

/*==============================================================*/
/* Table: ContractAttachment                                    */
/*==============================================================*/
 create table  ContractAttachment (
    AttachmentId            INTEGER              PRIMARY KEY AUTOINCREMENT,
    Name                    varchar(120)                   not null, 
    Url                     varchar(255)                       null, 
    Data                    blob                               null, 
    ExternalId              text(20)                           null, 
    LastUpdate              datetime                           null 
)
go

/*==============================================================*/
/* Table: ContractAttachmentLink                                */
/*==============================================================*/
create table  ContractAttachmentLink ( 
    AttachmentId          numeric(9)                     not null, 
    ContractId            numeric(9)                     not null, 
    constraint PK_CONTRACTATTACHMENTLINK primary key (AttachmentId,ContractId) 
 )
go

/*==============================================================*/
/* Table: Currency - definicja walut                            */
/*==============================================================*/
create table Currency (
   CurrencyId           INTEGER              PRIMARY KEY AUTOINCREMENT,
   IsActive             smallint             not null,
   Name                 text(10)             not null,
   Code                 text(10)             not null,
   Rate                 decimal(14,4)        not null,
   IsPayoutAllowed      smallint             null,
   IsDefault            smallint             null,
   ExternalId           text(20)             null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null
)
go

/*==============================================================*/
/* Index: CurrencyExtIdInx                                      */
/*==============================================================*/
create index CurrencyExtIdInx on Currency (
ExternalId ASC,
IsActive ASC
)
go

/*==============================================================*/
/* Index: CurrencyNameInx                                       */
/*==============================================================*/
create index CurrencyNameInx on Currency (
Name ASC,
IsActive ASC
)
go

/*==============================================================*/
/* Index: CurrencyCode                                          */
/*==============================================================*/
create index CurrencyCode on Currency (
Code ASC
)
go

/*==============================================================*/
/* Table: Customer - kontrahenci                                */
/*==============================================================*/
create table Customer (
   CustomerId           INTEGER              PRIMARY KEY AUTOINCREMENT,
   LevelNumber          int                  null,
   IsActive             smallint             not null,
   Name                 text(255)            not null,
   AbbrevName           text(40)             null,
   Town                 text(120)            null,
   PostalCode           text(20)             null,
   PostalName           text(120)            null,
   Street               text(120)            null,
   House                text(20)             null,
   ApartmentNo          text(10)             null,
   BankName             text(255)            null,
   BankAccount          text(40)             null,
   Nip                  text(20)             null,
   Phone                text(20)             null,
   Email                text(120)            null,
   LoyaltyPoints        decimal(14,4)        null,
   DiscountRate         decimal(6,2)         null,
   DefaultPaymentForm   smallint             null,
   DefaultPriceLevel    smallint             null,
   DefaultPaymentTerm   smallint             null,
   ShopNumber           int                  null,
   IsRetailCustomer     smallint             null,
   IsCustomer           smallint             null,
   IsManufacturer       smallint             null,
   IsSupplier           smallint             null,
   IsShop               smallint             null,
   IsCentralStore       smallint             null,
   Externalid           text(20)             null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null,
   EdiFormat            text(10)             null,
   SyncLastUpdate       datetime             null,
   CountryId            int                  null,
   Wojewodztwo          text(30)             null,
   Powiat               text(30)             null,
   Gmina                text(30)             null,
   USCode               text(20)             null,
   VatPayerStatus       varchar(255)         null,
   VatPayerStatusDate   datetime             null
)
go

/*==============================================================*/
/* Index: CustomerExtId                                         */
/*==============================================================*/
create index CustomerExtId on Customer (
Externalid ASC,
IsActive ASC
)
go

/*==============================================================*/
/* Index: CustomerName                                          */
/*==============================================================*/
create index CustomerName on Customer (
Name ASC,
IsActive ASC
)
go

/*==============================================================*/
/* Index: CustomerNip                                           */
/*==============================================================*/
create index CustomerNip on Customer (
Nip ASC,
IsActive ASC
)
go

/*==============================================================*/
/* Index: CustomerLastUpdateInx                                 */
/*==============================================================*/
create index CustomerLastUpdateInx on Customer (
LastUpdate ASC
)
go

/*==============================================================*/
/* Table: CustomerCard                                          */
/*==============================================================*/
create table CustomerCard (
   CustomerCardId       INTEGER              PRIMARY KEY AUTOINCREMENT,
   IsActive             smallint             not null,
   CustomerId           numeric(9)           not null,
   CardCode             text(40)             not null,
   CardType             smallint             default 0 null,
   IsSuspended          smallint             not null,
   ValidFrom            datetime             null,
   ValidUntil           datetime             null,
   AccountId            integer              null,
   Invalidated          datetime             null,
   Owner                varchar(40)          null,
   Desc1                varchar(40)          null,
   Desc2                varchar(40)          null,
   ExternalId           varchar(20)          null
)
go

/*==============================================================*/
/* Index: CustomerCardInx                                       */
/*==============================================================*/
create index CustomerCardInx on CustomerCard (
CardCode ASC,
IsActive ASC
)
go

/*==============================================================*/
/* Table: CustomerAccount                                       */
/*==============================================================*/

create table CustomerAccount (
   AccountId            INTEGER              PRIMARY KEY AUTOINCREMENT,
   CustomerId           INTEGER              not null,
   Type                 smallint             null,
   IsActive             smallint             null,
   Name                 varchar(40)          null,
   InitialBalance       decimal(15,4)        null,
   Balance              decimal(15,4)        null,
   LastUpdateBalance    datetime             not null,
   LastUpdate           datetime             not null,
   ExternalId           varchar(40)          null
)
go

create index CustomerAccountExtId on CustomerAccount (ExternalId)
go


/*==============================================================*/
/* Table: CustomerDescription                                   */
/*==============================================================*/
create table CustomerDescription (
   CustomerId            numeric(9)                     not null,
   DescType             smallint                       not null,
   Ordinal              smallint                       not null,
   Description          text(255)                      null,
   constraint PK_CUSTOMERDESCRIPTION primary key (CustomerId, DescType, Ordinal)
)
go

CREATE INDEX CustomerDescription_customerId_inx ON CustomerDescription (CustomerId ASC)
go


/*==============================================================*/
/* Table: DeviceConfig                                          */
/*==============================================================*/
create table DeviceConfig (
   PosId                numeric(9)                     not null,
   ParamGroup           text(255)                      not null,
   ParamName            text(255)                      not null,
   Ordinal              int                            not null,
   ParamValue           text(255)                      null,
   Creation             datetime                       null,
   LastUpdate           datetime                       null,
   constraint PK_DEVICECONFIG primary key (PosId, ParamGroup, ParamName, Ordinal)
)
go

/*==============================================================*/
/* Table: DiscountLevel                                         */
/*==============================================================*/
create table DiscountLevel (
   LevelNumber          int                  not null,
   DiscountRate         decimal(6,2)         not null,
   IsActive             smallint             not null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null,
   Name                 text(40)             not null,
   constraint PK_DISCOUNTLEVEL primary key (LevelNumber)
)
go

/*==============================================================*/
/* Table: DiscountRule                                          */
/*==============================================================*/
create table DiscountRule (
   DiscountRuleId       INTEGER              PRIMARY KEY AUTOINCREMENT,
   RulesGroupId         numeric(9)           not null,
   Name                 text(40)             null,
   Priority             int                  null,
   Type                 text(5)              null,
   Scope                text(5)              null,
   DiscountedProductCode text(40)             null,
   TriggerProductCode   text(40)             null,
   DiscountType         text(5)              null,
   IsCross              smallint             null
)
go

/*==============================================================*/
/* Table: DiscountRulesGroup                                    */
/*==============================================================*/
create table DiscountRulesGroup (
   RulesGroupId         INTEGER              PRIMARY KEY AUTOINCREMENT,
   DateAndTime          datetime             null,
   IsActive             smallint             null,
   ErrorCode            int                  null,
   ErrorDesc            text(80)             null
)
go

/*==============================================================*/
/* Table: FiscalReport                                          */
/*==============================================================*/
create table FiscalReport (
   PosId                numeric(9)           not null,
   FiscalReportId       INTEGER              PRIMARY KEY AUTOINCREMENT,
   OperatorId           numeric(9)           null,
   ReportDate           datetime             null,
   IsActive             smallint             null,
   Type                 int                  null,
   FiscalReportNumber   int                  null,
   ReceiptCount         int                  null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null,
   ExternalId           text(20)             null,
   CanceledReceiptCount int                  null,
   CanceledReceiptValue decimal(14,2)        null,
   UniqueFiscalNumber   text(20)             null,
   DbChangesCount       int                  null,
   PrintoutCount        int                  null,
   MemoryResetCount     int                  null,
   LastReceiptNo        int                  null,
   LastPrintoutNo       int                  null
)
go

/*==============================================================*/
/* Index: FiscalReportExternalIdInx                             */
/*==============================================================*/
create index FiscalReportExternalIdInx on FiscalReport ( ExternalId )
go

/*==============================================================*/
/* Index: FiscalReportDateTimeInx                               */
/*==============================================================*/
create index FiscalReportDateTimeInx on FiscalReport (
ReportDate ASC,
PosId ASC
)
go

/*==============================================================*/
/* Table: FiscalReportVat - raporty dobowe                      */
/*==============================================================*/
create table FiscalReportVat (
   PosId                numeric(9)           not null,
   FiscalReportId       numeric(9)           not null,
   Type                 smallint             not null,
   VatRateValue         decimal(6,2)         not null,
   Brutto               decimal(14,2)        null,
   Vat                  decimal(14,2)        null,
   constraint PK_FISCALREPORTVAT primary key (FiscalReportId, PosId, Type, VatRateValue)
)
go

/*==============================================================*/
/* Table: GastroElement                                         */
/*==============================================================*/
create table GastroElement (
   GastroSetProductId   numeric(9)                     not null,
   Ordinal              smallint                       not null,
   Type                 smallint                       null,
   Name                 varchar(60)                    null,
   Price                decimal(15,4)                  null,
   Quantity             decimal(15,4)                  null,
   IsDifferentPrice     smallint                       null,
   GastroElementOrdinal smallint                       null,
   constraint PK_GASTROELEMENT primary key (GastroSetProductId, Ordinal)
)
go

/*==============================================================*/
/* Table: GastroElementList                                     */
/*==============================================================*/
create table GastroElementList (
   GastroSetProductId   numeric(9)                     not null,
   Ordinal              smallint                       not null,
   ListOrdinal          smallint                       not null,
   ElementProductId     numeric(9)                     not null,
   Price                decimal(15,4)                  null,
   Quantity             decimal(15,4)                  null,
   IsDefault            smallint                       null,
   IsPriceFromProduct   smallint                       null,
   constraint PK_GASTROELEMENTLIST primary key (GastroSetProductId, Ordinal, ListOrdinal)
)
go

/*==============================================================*/
/* Table: Ingredient                                            */
/*==============================================================*/
create table Ingredient (
   PacketId             numeric(9)           not null,
   IngredientId         numeric(9)           not null,
   Type                 smallint             not null,
   Quantity             decimal(14,4)        null,
   Value                decimal(14,2)        null,
   constraint PK_INGREDIENT primary key (PacketId, IngredientId, Type)
)
go

CREATE INDEX ingredient_packetid_inx ON ingredient (packetid ASC)
go

/*==============================================================*/
/* Table: LastUpdate                                            */
/*==============================================================*/
create table LastUpdate (
   UpdateName           text(40)             not null,
   UpdateTime           datetime             not null,
   constraint PK_LASTUPDATE primary key (UpdateName)
)
go

/*==============================================================*/
/* Table: LastUpdateNotes                                       */
/*==============================================================*/
create table LastUpdateNotes (
   UpdateName           text(40)             not null,
   UpdateValue          text(40)             not null,
   constraint PK_LASTUPDATENOTES primary key (UpdateName, UpdateValue)
)
go
/*==============================================================*/
/* Table: LotClear                                              */
/*==============================================================*/
create table LotClear (
   LotId                numeric(9)                     not null,
   PosId                numeric(9)                     not null,
   ReceiptItemId        numeric(9)                     not null,
   ReceiptId            numeric(9)                     not null,
   Type                 smallint                       not null,
   Quantity             decimal(14,4)                  null,
   Date                 datetime                       null,
   LastUpdate           datetime                       null,
   constraint PK_LOTCLEAR primary key (LotId, PosId, ReceiptItemId, ReceiptId, Type)
)
go

/*==============================================================*/
/* Index: LotClearUpdateInx                                     */
/*==============================================================*/
create index LotClearUpdateInx on LotClear (
LastUpdate
)
go

/*==============================================================*/
/* Table: LotStore                                              */
/*==============================================================*/
create table LotStore (
   LotId                INTEGER              PRIMARY KEY AUTOINCREMENT,
   ProductId            numeric(9)                     null,
   StoreId              numeric(9)                     null,
   PosId                numeric(9)                     null,
   ReceiptItemId        numeric(9)                     null,
   ReceiptId            numeric(9)                     null,
   Status               smallint                       null,
   Date                 datetime                       null,
   DocNr                varchar(60)                    null,
   LotNr                varchar(60)                    null,
   ExpireDate           datetime                       null,
   ProductionDate       datetime                       null,
   ProductionNr         varchar(60)                    null,
   PurchasePrice        decimal(14,2)                  null,
   VatRate              decimal(6,2)                   null,
   Quantity             decimal(14,4)                  null,
   Remaining            decimal(14,4)                  null,
   Reserved             decimal(14,4)                  null,
   LastUpdate           datetime                       null,
   ExternalId           varchar(20)                    null
)
go

/*==============================================================*/
/* Index: LotLastUpdateInx                                      */
/*==============================================================*/
create index LotLastUpdateInx on LotStore (
LastUpdate
)
go

/*==============================================================*/
/* Index: LotExtInx                                             */
/*==============================================================*/
create index LotExtInx on LotStore (
ExternalId
)
go

/*==============================================================*/
/* Index: LotNrInx                                              */
/*==============================================================*/
create index LotNrInx on LotStore (
LotNr
)
go

/*==============================================================*/
/* Index: LotDateInx                                            */
/*==============================================================*/
create index LotDateInx on LotStore (
ProductId,
Date
)
go

/*==============================================================*/
/* Table: MessageRule                                           */
/*==============================================================*/
create table MessageRule (
   MessageRuleId        INTEGER              PRIMARY KEY AUTOINCREMENT,
   MessageRulesGroupId  numeric(9)           null,
   Code                 text(5)              null,
   MessageText          text(255)            null
)
go

/*==============================================================*/
/* Table: MessageRulesGroup                                     */
/*==============================================================*/
create table MessageRulesGroup (
   MessageRulesGroupId  INTEGER              PRIMARY KEY AUTOINCREMENT,
   DateAndTime          datetime             null,
   IsActive             smallint             null,
   ErrorCode            int                  null,
   ErrorDesc            text(80)             null
)
go

/*==============================================================*/
/* Table: Operator - uzytkownicy                                */
/*==============================================================*/
create table Operator (
   OperatorId           INTEGER              PRIMARY KEY AUTOINCREMENT,
   ProfileId            numeric(9)           not null,
   PosId                numeric(9)           null,
   IsActive             smallint             not null,
   Number               int                  not null,
   FirstName            text(120)            null,
   FamilyName           text(255)            null,
   Description          text(255)            null,
   IdCardCode           text(100)            null,
   ExternalId           text(20)             null,
   CashLimit            decimal(14,2)        null,
   Password             text(60)             not null,
   LoginName            text(60)             not null,
   IsOffLine            smallint             not null,
   PrepaidUser          text(40)             null,
   PrepaidPassword      text(40)             null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null
)
go

/*==============================================================*/
/* Index: OperatorExtIdInx                                      */
/*==============================================================*/
create index OperatorExtIdInx on Operator (
ExternalId ASC,
IsActive ASC
)
go

/*==============================================================*/
/* Index: OperatorLoginInx                                      */
/*==============================================================*/
create index OperatorLoginInx on Operator (
LoginName ASC
)
go

/*==============================================================*/
/* Table: OperatorConfig                                        */
/*==============================================================*/
create table OperatorConfig (
   OperatorId           numeric(9)           not null,
   ParamGroup           text(255)            not null,
   ParamName            text(255)            not null,
   Ordinal              int                  not null,
   ParamValue           text(255)            not null,
   LastUpdate           datetime             not null,
   constraint PK_OPERATORCONFIG primary key (OperatorId, ParamGroup, ParamName, Ordinal)
)
go

/*==============================================================*/
/* Table: OperatorProfile                                       */
/*==============================================================*/
create table OperatorProfile (
   ProfileId            INTEGER              PRIMARY KEY AUTOINCREMENT,
   Type                 int                  not null,
   Name                 text(40)             not null,
   ExternalId           text(20)             null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null
)
go

/*==============================================================*/
/* Index: OperProfileExternalIdInx                              */
/*==============================================================*/
create index OperProfileExternalIdInx on OperatorProfile (
ExternalId ASC
)
go

/*==============================================================*/
/* Table: OperatorSession                                       */
/*==============================================================*/
create table OperatorSession (
   OperatorId           numeric(9)           not null,
   ShiftId              numeric(9)           not null,
   SessionId            INTEGER              PRIMARY KEY AUTOINCREMENT,
   PosId                numeric(9)           null,
   Logon                datetime             null,
   Logoff               datetime             null,
   AppType              varchar(30)          null,
   AppVersion           varchar(20)          null,
   LastUpdate           datetime             null
)
go

/*==============================================================*/
/* Table: OperatorShift                                         */
/*==============================================================*/
create table OperatorShift (
   OperatorId           numeric(9)           not null,
   ShiftId              INTEGER              PRIMARY KEY AUTOINCREMENT,
   ShiftStart           datetime             null,
   ShiftStop            datetime             null,
   ShiftStatus          int                  not null,
   Type                 int                  not null,
   ExternalId           text(20)             null,
   LastUpdate           datetime             null
)
go

/*==============================================================*/
/* Index: ShiftStatusInx                                        */
/*==============================================================*/
create index ShiftStatusInx on OperatorShift (
ShiftStatus ASC,
OperatorId ASC
)
go

/*==============================================================*/
/* Index: ShiftUntilInx                                         */
/*==============================================================*/
create index ShiftUntilInx on OperatorShift (
OperatorId ASC,
ShiftStop ASC
)
go

/*==============================================================*/
/* Index: OperShiftExtIdInx                                     */
/*==============================================================*/
create index OperShiftExtIdInx on OperatorShift (
OperatorId ASC,
ExternalId ASC
)
go

/*==============================================================*/
/* Table: PaymentForm                                           */
/*==============================================================*/
create table PaymentForm (
   PaymentFormId        INTEGER              PRIMARY KEY AUTOINCREMENT,
   IsActive             smallint             not null,
   Name                 text(60)             not null,
   Type                 int                  not null,
   CardPaymentService   text(40)             null,
   IsChangeAllowed      smallint             not null,
   ExternalId           text(20)             null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null
)
go

/*==============================================================*/
/* Index: PaymentFormExtIdInx                                   */
/*==============================================================*/
create index PaymentFormExtIdInx on PaymentForm (
ExternalId ASC,
IsActive ASC
)
go

/*==============================================================*/
/* Index: PaymentFormNameInx                                    */
/*==============================================================*/
create index PaymentFormNameInx on PaymentForm (
Name ASC,
IsActive ASC
)
go

/*==============================================================*/
/* Table: PaymentPlan                                           */
/*==============================================================*/
create table PaymentPlan (
   PosId                numeric(9)           not null,
   ReceiptId            numeric(9)           not null,
   Ordinal              int                  not null,
   Type                 int                  not null,
   PaymentDate          datetime             not null,
   PaymentFormType      int                  not null,
   Completed            smallint             not null,
   constraint PK_PAYMENTPLAN primary key (ReceiptId, PosId, Ordinal, Type)
)
go

/*==============================================================*/
/* Table: PointsRule                                            */
/*==============================================================*/
create table PointsRule (
   RuleId               INTEGER              PRIMARY KEY AUTOINCREMENT,
   Name                 text(40)             null,
   Branch               smallint             default 0 null,
   Type                 smallint             null,
   Priority             smallint             null,
   Date1                datetime             null,
   Date2                datetime             null,
   Number1              decimal(16,4)        null,
   Number2              decimal(16,4)        null,
   Number3              decimal(16,4)        null,
   Number4              decimal(16,4)        null,
   Number5              decimal(16,4)        null,
   Int1                 int                  null,
   Int2                 int                  null,
   Int3                 int                  null,
   Int4                 int                  null,
   Int5                 int                  null,
   CardType             smallint             null,
   Str1                 varchar(128)         null,
   Str2                 varchar(128)         null,
   IsActive             smallint             not null,
   ExternalId           text(20)             null,
   Creation             datetime             null,
   LastUpdate           datetime             null
)
go


/*==============================================================*/
/* Table: PointsRuleParam                                            */
/*==============================================================*/
create table PointsRuleParam (
   ParamId              INTEGER              PRIMARY KEY AUTOINCREMENT,
   RuleId               numeric(9)                     not null,
   ParamType            int                            null,
   Param1               int                            null,
   Param2               varchar(60)                    null
)
go


/*==============================================================*/
/* Table: PointsRuleProduct                                     */
/*==============================================================*/
create table PointsRuleProduct (
   RuleId               numeric(9)                     not null,
   ProductId            numeric(9)                     not null,
   Type                 int                            not null,
   Quantity             decimal(15,4)                  null,
   Price                decimal(16,4)                  null,
   Param1               int                            null,
   constraint PK_POINTSRULEPRODUCT primary key (RuleId, ProductId, Type)
)
go

/*==============================================================*/
/* Table: PointsRuleProductCode                                 */
/*==============================================================*/
create table PointsRuleProductCode (
   RuleId               numeric(9)                     not null,
   Code                 varchar(40)                     not null,
   Type                 int                            not null,
   Quantity             decimal(15,4)                  null,
   Price                decimal(16,4)                  null,
   Param1               int                            null,
   constraint PK_POINTSRULEPRODUCTCODE primary key (RuleId, Code, Type)
)
go

/*==============================================================*/
/* Table: Pos                                                   */
/*==============================================================*/
create table Pos (
   PosId                INTEGER              PRIMARY KEY AUTOINCREMENT,
   PosGroupId           numeric(9)           not null,
   Type                 int                  null,
   Name                 text(40)             null,
   Number               int                  not null,
   IsActive             smallint             not null,
   ExternalId           text(20)             null,
   Creation             datetime             null,
   LastUpdate           datetime             null
)
go

/*==============================================================*/
/* Index: PosNameInx                                            */
/*==============================================================*/
create index PosNameInx on Pos (
Name ASC
)
go

/*==============================================================*/
/* Table: PosConfigParam                                        */
/*==============================================================*/
create table PosConfigParam (
   PosGroupId           numeric(9)           not null,
   ParamGroup           text(255)            not null,
   ParamName            text(255)            not null,
   Ordinal              int                  not null,
   ParamValue           text(255)            null,
   Creation             datetime             null,
   LastUpdate           datetime             null,
   constraint PK_POSCONFIGPARAM primary key (ParamGroup, ParamName, PosGroupId, Ordinal)
)
go

/*==============================================================*/
/* Table: PosDocCounter                                         */
/*==============================================================*/
create table PosDocCounter (
   Year                 int                  not null,
   Month                int                  not null,
   DocType              int                  not null,
   Counter              int                  not null,
   constraint PK_POSDOCCOUNTER primary key (Year, Month, DocType)
)
go

/*==============================================================*/
/* Table: PosGroup                                              */
/*==============================================================*/
create table PosGroup (
   PosGroupId           INTEGER              PRIMARY KEY AUTOINCREMENT,
   Name                 text(40)             null,
   IsActive             smallint             null,
   Creation             datetime             null,
   LastUpdate           datetime             null,
   ExternalId           text(20)             null
)
go


/*==============================================================*/
/* Index: PosGroupExternalIdInx                                 */
/*==============================================================*/
create index PosGroupExternalIdInx on PosGroup ( ExternalId )
go

/*==============================================================*/
/* Table: PosProductExport                                      */
/*==============================================================*/
create table PosProductExport (
   PosId                numeric(9)                     not null,
   ProductId            numeric(9)                     not null,
   Barcode              text(40)                       not null,
   Type                 smallint                       not null,
   Plu                  integer                        null,
   Price                decimal(14,2)                  null,
   ExportCmd            smallint                       null,
   ExportStatus         smallint                       null,
   Creation             datetime                       null,
   LastUpdate           datetime                       null,
   constraint PK_POSPRODUCTEXPORT primary key (PosId, ProductId, Barcode, Type)
)
go

/*==============================================================*/
/* Table: PosProductGroup                                       */
/*==============================================================*/
create table PosProductGroup (
   PosId                numeric(9)           not null,
   GroupId              numeric(9)           not null,
   constraint PK_POSPRODUCTGROUP primary key (PosId, GroupId)
)
go

/*==============================================================*/
/* Table: PricePlan                                             */
/*==============================================================*/
create table PricePlan (
   PlanId               INTEGER              PRIMARY KEY AUTOINCREMENT,
   ValidFrom            datetime             not null,
   ValidUntil           datetime             not null,
   PlanType             int                  not null,
   IsActive             smallint             null,
   Creation             datetime             null,
   LastUpdate           datetime             null,
   ExternalId           text(20)             null
)
go

/*==============================================================*/
/* Index: PricePlanValidIndex                                   */
/*==============================================================*/
create index PricePlanValidIndex on PricePlan (
ValidFrom ASC,
ValidUntil ASC
)
go

/*==============================================================*/
/* Index: PricePlanExtId_inx                                    */
/*==============================================================*/
create index PricePlanExtId_inx on PricePlan (
ExternalId ASC
)
go

/*==============================================================*/
/* Table: PricePlanProduct                                      */
/*==============================================================*/
create table PricePlanProduct (
   PlanId               decimal(9)           not null,
   ProductId            numeric(9)           not null,
   PriceLevel           smallint             not null,
   Price                decimal(14,2)        not null,
   constraint PK_PRICEPLANPRODUCT primary key (PlanId, ProductId, PriceLevel)
)
go

/*==============================================================*/
/* Table: Product - Towary                                      */
/*==============================================================*/
create table Product (
   ProductId            INTEGER              PRIMARY KEY AUTOINCREMENT,
   IsActive             smallint             not null,
   GroupId              numeric(9)           not null,
   UnitNameId           numeric(9)           not null,
   PackageId            numeric(9)           null,
   VatRateId            numeric(9)           not null,
   ManufacturerId       numeric(9)           null,
   Name                 text(120)            not null,
   AbbrevName           text(40)             null,
   Price1               decimal(14,2)        not null,
   Price2               decimal(14,2)        null,
   Price3               decimal(14,2)        null,
   Price4               decimal(14,2)        null,
   PriceMin             decimal(14,2)        null,
   PricePurchaseNetto   decimal(14,2)        null,
   PriceForUnit         decimal(14,2)        null,
   ProductType          int                  not null,
   Pkwiu                text(40)             null,
   PriceModifyLevel     int                  null,
   IsExisable           smallint             null,
   AdditionalFiscalPrinter smallint          null,
   KitchenPrinterNumber int                  null,
   KitchenPrinterNumbers text(40)            null,
   ExternalId           text(20)             null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null,
   Tara                 decimal(15,3)        null,
   CustomerAge          int                  null,
   QuantityRequired     smallint             null,
   WeightRequired       smallint             null,
   AdditionalDiscount   decimal(5,3)         null,
   IsFuel               smallint             null,
   LotControl           smallint             null,
   IsPending            smallint             null,
   SerialNumberRequired  smallint            null,
   IsLocalProduct       smallint             null,
   PackageMultiplier    decimal(15,4)        null,
   PressIndex 	        varchar(20)          null,
   PressMutation        varchar(20)          null,
   SyncLastUpdate       datetime             null,
   Indeks               varchar(40)          null,
   HideOnPos            smallint             null,
   Favorite             smallint             null,
   ImageLink            text(255)            null,
   Precission           smallint             null,
   SoldAmount           decimal(14,3)        null,
   SoldValue            decimal(14,2)        null,
   StockAmount          decimal(14,3)        null,
   StockBlocked         decimal(14,3)        null,
   IsSplitPayment       smallint             null,
   GTU                  text(5)              null,
   FrakId               smallint             null
)
go

/*==============================================================*/
/* Index: ProductExtId                                          */
/*==============================================================*/
create index ProductExtId on Product (
ExternalId ASC,
IsActive ASC
)
go

/*==============================================================*/
/* Index: ProductName                                           */
/*==============================================================*/
create index ProductName on Product (
Name ASC,
IsActive ASC
)
go

/*==============================================================*/
/* Index: ProductLastUpdateInx                                  */
/*==============================================================*/
create index ProductLastUpdateInx on Product (
LastUpdate ASC
)
go

/* Index: ProductId                                 		    */
/*==============================================================*/
create index ProductId on Product (
ProductId ASC
)
go

/*==============================================================*/
/* Table: ProductCategoryValue                                  */
/*==============================================================*/
create table ProductCategoryValue (
   ProductId            numeric(9)           not null,
   CategoryValueId      numeric(9)           not null,
   constraint PK_PRODUCTCATEGORYVALUE primary key (ProductId, CategoryValueId)
)
go

CREATE INDEX productcategoryvalue_productid_inx ON productcategoryvalue (productid ASC)
go

/*==============================================================*/
/* Table: ProductDescription                                    */
/*==============================================================*/
create table ProductDescription (
   ProductId            numeric(9)                     not null,
   DescType             smallint                       not null,
   Ordinal              smallint                       not null,
   Description          text(255)                      null,
   constraint PK_PRODUCTDESCRIPTION primary key (ProductId, DescType, Ordinal)
)
go

CREATE INDEX productdescription_productid_inx ON productdescription (productid ASC)
go

/*==============================================================*/
/* Table: ProductGroup                                          */
/*==============================================================*/
create table ProductGroup (
   GroupId              INTEGER              PRIMARY KEY AUTOINCREMENT,
   IsActive             smallint             not null,
   Name                 text(40)             not null,
   GroupType            int                  null,
   GroupLevel           int                  null,
   CustomerAge          int                  null,
   QuantityRequired     smallint             null,
   WeightRequired       smallint             null,
   ExternalId           text(20)             null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null,
   SerialNumberRequired smallint             null,
   HiddenOnPanels       smallint             null,
   HiddenOnBiller       smallint             null,
   IsExisable           smallint             null  
)
go

/*==============================================================*/
/* Index: ProductGroupExtId                                     */
/*==============================================================*/
create index ProductGroupExtId on ProductGroup (
ExternalId ASC,
IsActive ASC
)
go

/*==============================================================*/
/* Table: ProductGroupTree                                      */
/*==============================================================*/
create table ProductGroupTree (
   GroupId              numeric(9)           not null,
   ChildGroupId         numeric(9)           not null,
   constraint PK_PRODUCTGROUPTREE primary key (GroupId, ChildGroupId)
)
go

/*==============================================================*/
/* Table: ProductImages                                         */
/*==============================================================*/
create table ProductImages (
   ImageId              INTEGER              PRIMARY KEY AUTOINCREMENT,
   Type                 int                  null,
   Image                text(254)            null,
   RelationId           numeric(9)           null,
   RelationType         int                  null,
   ExternalId           text(20)             null
)
go

/*==============================================================*/
/* Index: ProductImgRelation                                    */
/*==============================================================*/
create index ProductImgRelation on ProductImages (
RelationType ASC,
RelationId ASC
)
go

/*==============================================================*/
/* Index: ProductImgExtId                                       */
/*==============================================================*/
create index ProductImgExtId on ProductImages (
ExternalId ASC
)
go

/*==============================================================*/
/* Table: ProductSet                                            */
/*==============================================================*/
create table ProductSet (
   MainProductId        numeric(9)           not null,
   SetType              int                  not null,
   SetSubtype           int                  not null,
   SetElementId         numeric(9)           null,
   constraint PK_PRODUCTSET primary key (MainProductId, SetType, SetSubtype)
)
go

CREATE INDEX productset_mainproductid_inx ON productset (mainproductid ASC)
go

/*==============================================================*/
/* Table: ProfileConfig                                         */
/*==============================================================*/
create table ProfileConfig (
   ProfileId            numeric(9)           not null,
   ParamGroup           text(255)            not null,
   ParamName            text(255)            not null,
   Ordinal              int                  not null,
   ParamValue           text(255)            null,
   LastUpdate           datetime             null,
   constraint PK_PROFILECONFIG primary key (ProfileId, ParamGroup, ParamName, Ordinal)
)
go

/*==============================================================*/
/* Table: RateLog                                               */
/*==============================================================*/
create table RateLog (
   CurrencyId           numeric(9)                     not null,
   Date                 datetime                       not null,
   SrcSellRate          decimal(15,4)                  null,
   SrcPurchRate         decimal(15,4)                  null,
   SellSpread           decimal(15,4)                  null,
   PurchSpread          decimal(15,4)                  null,
   SellRate             decimal(15,4)                  null,
   PurchRate            decimal(15,4)                  null,
   SellTableNumber      varchar(60)                    null,
   PurchTableNumber     varchar(60)                    null,
   constraint PK_RATELOG primary key (CurrencyId, Date)
)
go

/*==============================================================*/
/* Table: Receipt - naglowki dokumentow                         */
/*==============================================================*/
create table Receipt (
   PosId                numeric(9)           not null,
   ReceiptId            INTEGER              PRIMARY KEY AUTOINCREMENT,
   OperatorId           numeric(9)           null,
   ShiftId              numeric(9)           null,
   CustomerId           numeric(9)           null,
   TaxPayerId           numeric(9)           null,
   StoreId              numeric(9)           null,
   IsActive             smallint             not null,
   IsOffline            smallint             null,
   Type                 int                  not null,
   ExtType              int                  default 0 null,
   VatMarkup            int                  default 0 null,
   DocCounter           int                  null,
   DocNr                text(60)             null,
   OperationDate        datetime             not null,
   CustomerCard         text(20)             null,
   CustomerId2Type      int                  null,
   CustomerId2          text(20)             null,
   PostalCode           text(10)             null,
   TotalBrutto          decimal(14,2)        null,
   TotalNetto           decimal(14,2)        null,
   TotalDiscount        decimal(14,2)        null,
   DepositSold          decimal(14,2)        null,
   DepositReturned      decimal(14,2)        null,
   TotalPrepaid         decimal(14,2)        null,
   TotalBill            decimal(14,2)        null,
   TotalToPay           decimal(14,2)        null,
   PrinterStatus        smallint             not null,
   PrinterNumber        int                  null,
   PrinterReportNumber  int                  null,
   PrintoutNumber       int                  null,
   ValueDiscount        decimal(14,2)        null,
   IsPending            smallint             null,
   PendingOwner         text(10)             null,
   DiscountRoundupMethod smallint            null,
   DocumentSource       smallint             null,
   PriceLevel           smallint             null, 
   ExternalId           text(20)             null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null,
   NielsenExported      smallint             default 0,
   NielsenStopExport    smallint             default 0
)
go

/*==============================================================*/
/* Index: ShiftIdOperatorIdInx                                       */
/*==============================================================*/
create index ShiftIdOperatorIdInx on Receipt (ShiftId,OperatorId)
go



/*==============================================================*/
/* Index: ReceiptDateInx                                        */
/*==============================================================*/
create index ReceiptDateInx on Receipt (
LastUpdate ASC
)
go

/*==============================================================*/
/* Index: ReceiptExtIdInx                                       */
/*==============================================================*/
create index ReceiptExtIdInx on Receipt (
PosId ASC,
ExternalId ASC
)
go

/*==============================================================*/
/* Table: ReceiptBill                                           */
/*==============================================================*/
create table ReceiptBill (
   BillId               INTEGER              PRIMARY KEY AUTOINCREMENT,
   ReceiptId            numeric(9)           not null,
   PosId                numeric(9)           not null,
   Value                decimal(14,2)        null,
   Interest             decimal(14,2)        null,
   Fee                  decimal(14,2)        null,
   Issuer               text(255)            null,
   Reason               text(255)            null,
   Barcode              text(128)            null
)
go

/*==============================================================*/
/* Table: ReceiptCoupon                                         */
/*==============================================================*/
create table ReceiptCoupon (
   PosId                numeric(9)           not null,
   ReceiptId            numeric(9)           not null,
   CouponId             INTEGER              PRIMARY KEY AUTOINCREMENT,
   CouponType           smallint             not null,
   CouponNumber         text(60)             null
)
go

/*==============================================================*/
/* Table: ReceiptCurrency                                       */
/*==============================================================*/
create table ReceiptCurrency (
   CurrencyId           numeric(9)           not null,
   PosId                numeric(9)           not null,
   ReceiptId            numeric(9)           not null,
   CurrencyRate         decimal(14,4)        null,
   Total                decimal(14,2)        null,
   TotalInCurrency      decimal(14,2)        null,
   constraint PK_RECEIPTCURRENCY primary key (ReceiptId, CurrencyId, PosId)
)
go

/*==============================================================*/
/* Table: ReceiptDescription                                    */
/*==============================================================*/
create table ReceiptDescription (
   ReceiptId            numeric(9)           not null,
   PosId                numeric(9)           not null,
   DescType             smallint             not null,
   Ordinal              smallint             not null,
   Description          varchar(255)         null,
   constraint PK_RECEIPTDESCRIPTION primary key (ReceiptId, PosId, DescType, Ordinal)
)
go

/*==============================================================*/
/* Table: ReceiptItem - pozycje dokumentow                      */
/*==============================================================*/
create table ReceiptItem (
   PosId                numeric(9)           not null,
   ReceiptItemId        INTEGER              PRIMARY KEY AUTOINCREMENT,
   ReceiptId            numeric(9)           not null,
   ProductId            numeric(9)           null,
   ExtReceiptItemId     numeric(9)           null,
   VatRate              decimal(6,2)         null,
   VatRateVMOrg         decimal(6,2)         null,
   ProductType          int                  null,
   BasePrice            decimal(14,2)        null,
   Price                decimal(14,2)        null,
   DiscountType         int                  null,
   DiscountSource       int                  null,
   DiscountValue        decimal(10,2)        null,
   Quantity             decimal(14,4)        null,
   TotalBrutto          decimal(14,2)        null,
   TotalNetto           decimal(14,2)        null,
   TotalDiscount        decimal(14,2)        null,
   IsStorno             smallint             null,
   IsReturn             smallint             null,
   Barcode              text(40)             null,
   ItemType             int                  null,
   RemainingQuantity    decimal(14,4)        null,
   SellPrice            decimal(14,2)        null,
   PriceMin             decimal(14,2)        null,
   DiscountedPrice      decimal(14,2)        null,
   CalcType             smallint             null,
   ExpireDate           datetime             null,
   ExternalId           text(20)             null,
   LotId                numeric(9)           null,
   LotNr                varchar(60)          null,
   PriceLevel           smallint             null
)
go

/*==============================================================*/
/* Index: ReceiptItemReceiptProductId                           */
/*==============================================================*/
create index ReceiptItemReceiptProductId on ReceiptItem (
ReceiptId ASC,
ProductId ASC
)
go

/*==============================================================*/
/* Index: ReceiptItemBarcodeProductIdInx                        */
/*==============================================================*/
create index  ReceiptItemBarcodeProductIdInx on ReceiptItem (Barcode, ProductId
)
go

/*==============================================================*/
/* Index: ReceiptItemExternalIdInx                              */
/*==============================================================*/
create index  ReceiptItemExternalIdInx on ReceiptItem (ExternalId)
go


/*==============================================================*/
/* Table: ReceiptItemIndex                                      */
/*==============================================================*/
create table ReceiptItemIndex (
   PosId                numeric(9)           not null,
   ReceiptItemId        numeric(9)           not null,
   ReceiptId            numeric(9)           not null,
   Type                 int                  not null,
   Ordinal              int                  not null,
   ItemIndex            text(40)             not null,
   constraint PK_RECEIPTITEMINDEX primary key (ReceiptId, ReceiptItemId, PosId, Type, Ordinal)
)
go

/*==============================================================*/
/* Table: ReceiptItemRule                                       */
/*==============================================================*/
create table ReceiptItemRule (
   RuleId               numeric(9)           not null,
   PosId                numeric(9)           not null,
   ReceiptItemId        numeric(9)           not null,
   ReceiptId            numeric(9)           not null,
   ReceiptPointsOrdinal smallint             DEFAULT 0 NOT NULL,
   Type                 smallint             not null,
   Number1              decimal(15,4)        not null,
   Number2              decimal(15,4)        NULL,
   Number3              decimal(15,4)        NULL,
   Number4              decimal(15,4)        NULL,
   Int1 			    smallint             NULL,
   Int2  			    smallint             NULL,
   Str1                 varchar(255)         not null,
   constraint PK_RECEIPTITEMRULE primary key (RuleId, ReceiptId, ReceiptItemId, PosId, ReceiptPointsOrdinal, Type)
)
go

/*==============================================================*/
/* Table: ReceiptPayment                                        */
/*==============================================================*/
create table ReceiptPayment (
   PaymentFormId        numeric(9)           not null,
   CurrencyId           numeric(9)           not null,
   PosId                numeric(9)           not null,
   ReceiptId            numeric(9)           not null,
   Value                decimal(14,2)        null,
   ValueInCurrency      decimal(14,2)        null,
   Change               decimal(14,2)        null,
   ChangeInCurrency     decimal(14,2)        null,
   Description          text(40)             null,
   constraint PK_RECEIPTPAYMENT primary key (ReceiptId, CurrencyId, PaymentFormId, PosId)
)
go

/*==============================================================*/
/* Table: ReceiptPoints                                         */
/*==============================================================*/
create table ReceiptPoints (
   ReceiptId            numeric(9)           not null,
   PosId                numeric(9)           not null,
   RuleId               numeric(9)           not null,
   Ordinal              smallint	     default 0,
   Type                 smallint	     default 0,
   Points               decimal(14,4)        null,
   Str1                 varchar(255)         null,
   Number_1             decimal(15,4)        null,
   Number_2             decimal(15,4)        null,
   Number_3             decimal(15,4)        null,
   Int_1                integer              null,
   Int_2                smallint             null,
   constraint PK_RECEIPTPOINTS primary key (ReceiptId, PosId, RuleId, Ordinal)
)
go

/*==============================================================*/
/* Table: ReceiptPointsParam                                    */
/*==============================================================*/
CREATE TABLE ReceiptPointsParam (
   ReceiptId            numeric(9)           NOT NULL,
   PosId                numeric(9)           NOT NULL,
   RuleId               numeric(9)           NOT NULL,
   ReceiptPointsOrdinal smallint	         NOT NULL,
   ParamType            smallint             NOT NULL,
   Param1               int                  NOT NULL,
   Param2               decimal(15,4)        NOT NULL,
   Param3               varchar(255)         NOT NULL,
   constraint PK_RECEIPTPOINTSPARAM primary key (ReceiptId, PosId, RuleId, ReceiptPointsOrdinal, ParamType)
)
go


/*==============================================================*/
/* Table: ReceiptRateLog                                        */
/*==============================================================*/
create table ReceiptRateLog (
   PosId                numeric(9)                     not null,
   ReceiptId            numeric(9)                     not null,
   CurrencyId           numeric(9)                     not null,
   Type                 int                            not null,
   Rate                 decimal(15,4)                  null,
   Date                 datetime                       null,
   TableNumber          varchar(60)                    null,
   constraint PK_RECEIPTRATELOG primary key (ReceiptId, PosId, CurrencyId, Type)
)
go

/*==============================================================*/
/* Table: ReceiptToReceipt                                      */
/*==============================================================*/
create table ReceiptToReceipt (
   ReceiptId            numeric(9)           not null,
   PosId                numeric(9)           not null,
   Type                 int                  not null,
   ReceiptId2           numeric(9)           not null,
   PosId2               numeric(9)           not null,
   constraint PK_RECEIPTTORECEIPT primary key (ReceiptId, PosId, Type,ReceiptId2,PosId2)
)
go

/*==============================================================*/
/* Table: ReceiptVat                                            */
/*==============================================================*/
create table ReceiptVat (
   ReceiptId            numeric(9)           not null,
   PosId                numeric(9)           not null,
   VatRate              decimal(6,2)         not null,
   Type                 int                  not null,
   Netto                decimal(14,2)        null,
   Vat                  decimal(14,2)        null,
   Brutto               decimal(14,2)        null,
   constraint PK_RECEIPTVAT primary key (ReceiptId, PosId, VatRate, Type)
)
go

/*==============================================================*/
/* Table: SessionEvent                                          */
/*==============================================================*/
create table SessionEvent (
   PosId                numeric(9)           not null,
   EventId              INTEGER              PRIMARY KEY AUTOINCREMENT,
   EventType            int                  not null,
   EventDescription     text(255)            null,
   EventParam           int                  null,
   EventTime            datetime             null
)
go

/*==============================================================*/
/* Table: Stock - stan towaru                                   */
/*==============================================================*/
create table Stock (
   StoreId              numeric(9)           not null,
   ProductId            numeric(9)           not null,
   Quantity             decimal(15,4)        null,
   Blocked              decimal(15,4)        null,
   Reserved             decimal(15,4)        null,
   Price                decimal(15,4)        null,
   LastUpdate           datetime             null,
   constraint PK_STOCK primary key (StoreId, ProductId)
)
go

/*==============================================================*/
/* Table: Store                                                 */
/*==============================================================*/
create table Store (
   StoreId              INTEGER              PRIMARY KEY AUTOINCREMENT,
   Name                 text(60)             not null,
   Number               int                  null,
   Creation             datetime             null,
   LastUpdate           datetime             null,
   ExternalId           text(20)             null
)
go

/*==============================================================*/
/* Index: StoreExternalIdInx                                    */
/*==============================================================*/
create index StoreExternalIdInx on Store ( ExternalId )
go

/*==============================================================*/
/* Table: StructureVersion                                      */
/*==============================================================*/
create table StructureVersion (
   Version              text(40)             not null,
   Status               int                  not null,
   constraint PK_STRUCTUREVERSION primary key (Version)
)
go

/*==============================================================*/
/* Table: Task                                                  */
/*==============================================================*/
create table Task (
   PosId                numeric(9)           not null,
   TaskId               INTEGER              PRIMARY KEY AUTOINCREMENT,
   TaskType             int                  null,
   Status               int                  null,
   Completed            smallint             null,
   ExternalId           text(20)             null,
   Creation             datetime             null,
   LastUpdate           datetime             null
)
go

/*==============================================================*/
/* Index: TaskCompletedInx                                      */
/*==============================================================*/
create index TaskCompletedInx on Task (
PosId ASC,
Completed ASC
)
go

/*==============================================================*/
/* Index: TaskExternalIdInx                                     */
/*==============================================================*/
create index TaskExternalIdInx on Task (
ExternalId ASC
)
go

/*==============================================================*/
/* Table: TaskParam                                             */
/*==============================================================*/
create table TaskParam (
   PosId                numeric(9)           not null,
   TaskId               numeric(9)           not null,
   ParamName            text(60)             not null,
   ParamValue           text(255)            null,
   constraint PK_TASKPARAM primary key (PosId, TaskId, ParamName)
)
go

/*==============================================================*/
/* Table: Threshold                                             */
/*==============================================================*/
create table Threshold (
   ThresholdId          INTEGER              PRIMARY KEY AUTOINCREMENT,
   DiscountRuleId       numeric(9)           null,
   DiscountValue        decimal(10,4)        null,
   FromValue            decimal(10,4)        null,
   ToValue              decimal(10,4)        null
)
go

/*==============================================================*/
/* Table: UnitName                                              */
/*==============================================================*/
create table UnitName (
   UnitNameId           INTEGER              PRIMARY KEY AUTOINCREMENT,
   IsActive             smallint             not null,
   Name                 text(10)             not null,
   Precission           int                  not null,
   ExternalId           text(20)             null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null
)
go

/*==============================================================*/
/* Index: UnitNameExtId                                         */
/*==============================================================*/
create index UnitNameExtId on UnitName (
ExternalId ASC
)
go

/*==============================================================*/
/* Table: VatRate                                               */
/*==============================================================*/
create table VatRate (
   VatRateId            INTEGER              PRIMARY KEY AUTOINCREMENT,
   IsActive             smallint             not null,
   Value                decimal(6,2)         not null,
   ExternalId           text(20)             null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null
)
go

/*==============================================================*/
/* Table: VatRateLtr                                            */
/* Odczytane z drukarki fiskalnej powiązanie literka-wartość    */
/* Value = -1:zwolniona, -2:niezdefiniowana, inne:wartość*100   */ 
/*==============================================================*/
create table VatRateLtr (
   Letter	         	text(1)              PRIMARY KEY,
   Value                decimal(6,2)         not null
)
go

/*==============================================================*/
/* Index: VatRateExtId                                          */
/*==============================================================*/
create index VatRateExtId on VatRate (
ExternalId ASC
)
go

/*==============================================================*/
/* Table:  DocExt                                               */
/*==============================================================*/
create table DocExt(
     PosId                numeric(9)                     not null,
     ReceiptId            numeric(9)                     not null,
     IssueDateTime        datetime                       null,
     DeliveryDateTime     datetime                       null,
     TotalRetailVat       decimal(14,2)                  null,
     ExternalDocNumber    text(60)                       null,
     Value1               decimal(15,3)                  null,
     Value2               decimal(15,3)                  null,
     Value3               decimal(15,3)                  null,
     Value4               decimal(15,3)                  null,
     Value5               decimal(15,3)                  null,
     constraint PK_DOCEXT primary key (PosId, ReceiptId)
)
go 

/*==============================================================*/
/* Table:  DocItemExt                                           */
/*==============================================================*/
create table DocItemExt(
     PosId                numeric(9)                     not null,
     ReceiptId            numeric(9)                     not null,
     ReceiptItemId        numeric(9)                     not null,
  	 OldRetailPriceBrutto decimal(14,2)                  null, 
	 OldRetailPriceNetto  decimal(14,2)                  null,
	 RetailPriceBrutto    decimal(14,2)                  null,
	 RetailPriceNetto      decimal(14,2)                 null, 
     CorrectedPurchasePriceBrutto decimal(14,2)          null, 
     CorrectedPurchasePriceNetto  decimal(14,2)          null,
     DefaultPurchasePriceNetto decimal(14,2)             null,
	 DefaultPurchasePriceBrutto decimal(14,2)            null,
     PurchasePriceNetto   decimal(14,2)                  null,
     PurchasePriceBrutto  decimal(14,2)                  null,
     OriginalTotalRetailBrutto decimal(14,2)             null,
     OriginalTotalRetailNetto  decimal(14,2)             null,
	 OriginalTotalPurchaseBrutto decimal(14,2)           null,
	 OriginalTotalPurchaseNetto  decimal(14,2)           null,
     OriginalQuantity  decimal(15,3)             null,
     CorrectedTotalRetailBrutto decimal(14,2)            null,
     CorrectedTotalRetailNetto  decimal(14,2)            null,
	 CorrectedTotalPurchaseBrutto decimal(14,2)          null,
	 CorrectedTotalPurchaseNetto  decimal(14,2)          null,
     IsActive         smallint                       not null, 
	 IsSourcePurchaseValueBrutto smallint            not null,
	 IsSourceRetailValueBrutto   smallint            not null,
	 IsSourcePriceValueQuantity  smallint            not null,
	 IsPriceRetailPurchase       smallint            not null,
     constraint PK_DOCITEMEXT primary key (ReceiptItemId, ReceiptId, PosId)
)
go 

/*==============================================================*/
/*==============================================================*/
create index ExternalDocNumberId on DocExt (
ExternalDocNumber
)
go

/*==============================================================*/
/*==============================================================*/
create table Airline
(
    LineId     INTEGER                 PRIMARY KEY AUTOINCREMENT,
    Name       varchar(255)            null,
    Code         varchar(16)           not null,
    IsActive     smallint              not null,
    ExternalId   text(20)              null,
    Creation      datetime             not null,
    LastUpdate      datetime           not null
)
go

create index idx_ArlineCode on Airline (Code asc)
go

create index idx_AirlineChange on Airline (LastUpdate desc)
go

/*==============================================================*/
/*==============================================================*/
create table Airport
(
    PortId      INTEGER              PRIMARY KEY AUTOINCREMENT,
    Name       varchar(255)                  null,
    Country        varchar(120)                  null,
    Code         varchar(16)           not null,
    DutyFree    smallint              not null,
    IsActive     smallint              not null,
    ExternalId   text(20)              null,
    Creation      datetime              not null,
    LastUpdate      datetime              not null
)
go

create index idx_AirportCode on Airport (Code asc)
go

create index idx_AirportChange on Airport (LastUpdate desc)
go

/*==============================================================*/
/* Table:  FiscalReportECopy                                           */
/*==============================================================*/
create table FiscalReportECopy(
	ReportId                        INTEGER         PRIMARY KEY AUTOINCREMENT,
	PosId                           numeric(9)      not null,
	LastUpdate                      datetime        not null,
	FormatForPharmacyPrinter        smallint        not null,
	SupportForPharmacyFunctions     smallint        not null,
	ReportDate                      datetime        not null,
	ReceiptCount                    numeric         not null,
	CancelledReceiptCount           numeric         not null,
	DbChangesCount                  numeric         not null,
	DailyReportNumber               numeric         not null,
	SaleBruttoValue                 decimal(20,2)   not null,
	PTUValue                        decimal(20,2)   not null,
	PrintoutCount                   numeric         not null,
	ResetCount                      numeric         not null,
	LastReceiptNumber               numeric         not null,
	LastPrintoutNumber              numeric         not null,
	ValuationCount                  numeric         not null,
	CancelledReceiptValue           decimal(20,2)   not null,
	Payment                         decimal(20,2)   not null,
	TotalizerA                      decimal(20,2)   not null,
	TotalizerB                      decimal(20,2)   not null,
	TotalizerC                      decimal(20,2)   not null,
	TotalizerD                      decimal(20,2)   not null,
	TotalizerE                      decimal(20,2)   not null,
	TotalizerF                      decimal(20,2)   not null,
	TotalizerG                      decimal(20,2)   not null,
	PTUA                            decimal(20,2)   not null,
	PTUB                            decimal(20,2)   not null,
	PTUC                            decimal(20,2)   not null,
	PTUD                            decimal(20,2)   not null,
	PTUE                            decimal(20,2)   not null,
	PTUF                            decimal(20,2)   not null,
	PTUG                            decimal(20,2)   not null,
	TotalValuePTU_A                 decimal(20,2)   not null,
	TotalValuePTU_B                 decimal(20,2)   not null,
	TotalValuePTU_C                 decimal(20,2)   not null,
	TotalValuePTU_D                 decimal(20,2)   not null,
	TotalValuePTU_E                 decimal(20,2)   not null,
	TotalValuePTU_F                 decimal(20,2)   not null,
	TotalValuePTU_G                 decimal(20,2)   not null,
	TotalValuePTU                   decimal(20,2)   not null,
	TotalSaleBrutto                 decimal(20,2)   not null,
	TotalizerA_FV                   decimal(20,2)   not null,
	TotalizerB_FV                   decimal(20,2)   not null,
	TotalizerC_FV                   decimal(20,2)   not null,
	TotalizerD_FV                   decimal(20,2)   not null,
	TotalizerE_FV                   decimal(20,2)   not null,
	TotalizerF_FV                   decimal(20,2)   not null,
	TotalizerG_FV                   decimal(20,2)   not null,
	PTUA_FV                         decimal(20,2)   not null,
	PTUB_FV                         decimal(20,2)   not null,
	PTUC_FV                         decimal(20,2)   not null,
	PTUD_FV                         decimal(20,2)   not null,
	PTUE_FV                         decimal(20,2)   not null,
	PTUF_FV                         decimal(20,2)   not null,
	PTUG_FV                         decimal(20,2)   not null,
	TotalValuePTU_A_FV              decimal(20,2)   not null,
	TotalValuePTU_B_FV              decimal(20,2)   not null,
	TotalValuePTU_C_FV              decimal(20,2)   not null,
	TotalValuePTU_D_FV              decimal(20,2)   not null,
	TotalValuePTU_E_FV              decimal(20,2)   not null,
	TotalValuePTU_F_FV              decimal(20,2)   not null,
	TotalValuePTU_G_FV              decimal(20,2)   not null,
	TotalValuePTU_FV                decimal(20,2)   not null,
	TotalSaleBruttoFV               decimal(20,2)   not null,
	InvoiceCount                    numeric         not null,
	LastInvoiceNumber               numeric         not null,
	TotalSaleFV                     numeric         not null,
	TotalPTU_FV                     decimal(20,2)   not null,
	RatePTU_A                       smallint        not null,
	RatePTU_B                       smallint        not null,
	RatePTU_C                       smallint        not null,
	RatePTU_D                       smallint        not null,
	RatePTU_E                       smallint        not null,
	RatePTU_F                       smallint        not null,
	RatePTU_G                       smallint        not null,
	CurrencyName                    text(20)        not null,
	CancelledValuationCount         numeric         not null,
	ShortcutFromElectronicCopyData  text(80)        not null,
	FirstSale                       datetime        not null,
	LastSale                        datetime        not null
)
go

/*==============================================================*/
/* Table: GastroTransition
/*==============================================================*/
create table GastroTransition(
   VatRateId            INTEGER              not null,
   ProductId            INTEGER              not null,
   Type                 int                  not null,
   constraint PK_GASTROTRANSITION primary key (VatRateId, ProductId, Type)
)
go

/*==============================================================*/
/* Table: CustomerCardFormat
/*==============================================================*/
CREATE TABLE CustomerCardFormat(
   CustomerCardFormatId             INTEGER              PRIMARY KEY AUTOINCREMENT,
   IsActive                         INTEGER              not null,
   Name                             text(120)            not null,
   CodeLength                       INTEGER              not null,
   PrefixLength                     INTEGER              not null,
   PrefixPosition                   INTEGER              not null,
   YearLength                       INTEGER              not null,
   CustomerCodeLength               INTEGER              not null,
   CustomerCodePosition             INTEGER              not null,
   DatePosition                     INTEGER              not null,
   DiscountIdPosition               INTEGER              not null,
   Prefix                           text(10)             not null,
   IsExpireDate                     INTEGER              not null,
   IsMonth                          INTEGER              not null,
   IsControlDigit                   INTEGER              not null,
   IsAutoRecognize                  INTEGER              not null,
   ExternalId						text(20)             null,
   Creation                         datetime             not null,
   LastUpdate                       datetime             not null
)
go

/*==============================================================*/
/* Table: DiscountDefined                                       */
/*==============================================================*/
create table DiscountDefined (
   DiscountDefinedId                INTEGER              PRIMARY KEY AUTOINCREMENT,
   IsActive                         smallint             not null,
   Type                             int                  not null,
   Name                             text(120)            not null,
   Purpose                          smallint             not null,
   DateFrom                         datetime             null,
   DateTo                           datetime             null,
   Days                             text(10)             not null,
   Hours                            text(1000)           not null,
   DefaultValue                     decimal(20,2)        not null,
   MaxValue                         decimal(20,2)        not null,
   IsEditable                       smallint             not null,
   IsObligatory                     smallint             not null,
   IsPriceModify                    smallint             not null,
   RestrictionType                  smallint             not null,
   PurchaseValueFrom                decimal(20,2)        not null,
   PurchaseValueTo                  decimal(20,2)        not null,
   ProductGroupId                   int                  null,
   Creation                         datetime             null,
   LastUpdate                       datetime             null,
   ExternalId                       text(20)             null
)
go

/*==============================================================*/
/* Table: Country
/*==============================================================*/
create table Country (
   CountryId            INTEGER             PRIMARY KEY AUTOINCREMENT,
   IsActive             smallint            not null,
   Name                 text(60)            not null,
   Code                 text(10)            not null,
   CurrencyId           int                 not null,
   ExternalId           text(20)            null,
   Creation             datetime            not null,
   LastUpdate           datetime            not null
)
go

/*==============================================================*/
/* Table: RadDiscountDefinition
/*==============================================================*/
create table RadDiscountDefinition (
   UniqueId            text(30)             PRIMARY KEY,
   ProductCode         text(30)             not null,
   Name                text(60)             not null,
   DateStart           datetime             not null,
   DateStop            datetime             not null,
   LimitRemaining      decimal(20,2)        not null,
   DiscountType        int                  not null,
   DiscountValue       decimal(20,2)        not null,
   LastUpdate          datetime             not null
)
go

/*==============================================================*/
/* Table: Image
/*==============================================================*/
CREATE TABLE Image (
     ImageId          INTEGER            PRIMARY KEY AUTOINCREMENT,
     Img              BLOB               NOT NULL,
     Size             INT                NOT NULL,
     Name             VARCHAR(255)       NOT NULL,
     Describe         VARCHAR(2000)      NOT NULL,
     IsActive         INT	               NOT NULL,
     ExternalId       VARCHAR(20)        NULL,
     GroupType        INT                NOT NULL,
     Creation         DATETIME           NOT NULL,
     LastUpdate       DATETIME           NOT NULL
)
go

/*==============================================================*/
/* Table: ImageParam
/*==============================================================*/
CREATE TABLE ImageParam (
     ImageParamId     INTEGER            PRIMARY KEY AUTOINCREMENT,
     ImageId          INTEGER            NOT NULL,
     ParamType        INT                NOT NULL,
     Param1           INT                NULL,
     Param2           VARCHAR(255)       NULL
)
go


/*==============================================================*/
/* Table: ImageRelation
/*==============================================================*/
CREATE TABLE ImageRelation (
     ImageRelationId  INTEGER            PRIMARY KEY AUTOINCREMENT,
     ImageId          INTEGER            NOT NULL,
     RelationId       INT                NOT NULL,
     RelationType     INT                NOT NULL
)
go


/*==============================================================*/
/* Table: ReceiptPaymentBon
/*==============================================================*/
CREATE TABLE ReceiptPaymentBon (
   PaymentFormId        NUMERIC(9)           NOT NULL,
   CurrencyId           NUMERIC(9)           NOT NULL,
   PosId                NUMERIC(9)           NOT NULL,
   ReceiptId            NUMERIC(9)           NOT NULL,
   OrderNumber          SMALLINT             NOT NULL,
   BonCode              VARCHAR(40)          NULL,
   ServiceName          VARCHAR(25)          NULL,
   TransactionId        VARCHAR(40)          NULL,
   BonType 			    SMALLINT             NULL,
   BonValue             DECIMAL(16,4)        NULL,
   TransactionTime 	    DATETIME             NULL,
   constraint PK_ReceiptPaymentBon PRIMARY KEY (ReceiptId, CurrencyId, PaymentFormId, PosId, OrderNumber)
)
GO


/*==============================================================*/
/* Table: Phrase
/*==============================================================*/
CREATE TABLE Phrase (
   PhraseId             INTEGER              PRIMARY KEY AUTOINCREMENT,
   IsActive             smallint             NOT NULL,
   Type                 int                  NOT NULL,
   Text                 text(255)            NOT NULL,
   PriceLevel           int                  NOT NULL,
   ExternalId           text(20)             NULL,
   Creation             datetime             NOT NULL,
   LastUpdate           datetime             NOT NULL
)
GO

/*==============================================================*/
/* Table: LogTypeDef                                            */
/*==============================================================*/
create table LogTypeDef (
   TypeId               INTEGER              PRIMARY KEY,
   TypeCategory         text(50),
   Type                 INT                  not null,
   TypeValue            text(255)            not null,
   LastUpdate           datetime
)
go

/*==============================================================*/
/* Table: LogOperation                                          */
/*==============================================================*/
create table LogOperation (
   OperationId          INTEGER              PRIMARY KEY AUTOINCREMENT,
   PosId                numeric(9)           not null,
   OperatorId           numeric(9),
   AppTypeId            numeric(9)           not null,
   ModuleTypeId         numeric(9)           not null,
   OperationTypeId      numeric(9)           not null,
   OperationTime        datetime             not null,
   AdditionalDesc       text(255),
   ModItemId            text(30),
   ModTableTypeId       numeric(9),
   LastUpdate           datetime
)
go

/*==============================================================*/
/* Table: LogValChangeNum                                       */
/*==============================================================*/
create table LogValChangeNum (
   ValChangeId          INTEGER              PRIMARY KEY AUTOINCREMENT,
   OperationId          numeric(9)           not null,
   ValChangeTypeId      numeric(9)           not null,
   ValueBefore          decimal(15,4)        not null,
   ValueAfter           decimal(15,4)        not null,
   LastUpdate           datetime
)
go

/*==============================================================*/
/* Table: LogValChangeTxt                                       */
/*==============================================================*/
create table LogValChangeTxt (
   ValChangeId          INTEGER              PRIMARY KEY AUTOINCREMENT,
   OperationId          numeric(9)           not null,
   ValChangeTypeId      numeric(9)           not null,
   ValueBefore          text(255)            not null,
   ValueAfter           text(255)            not null,
   LastUpdate           datetime
)
go

/*==============================================================*/
/* Table: CodeDefinition                                        */
/*==============================================================*/
create table CodeDefinition (
   CodeDefId            INTEGER              PRIMARY KEY AUTOINCREMENT,
   Name                 text(255)            NOT NULL,
   IsActive             SMALLINT             NOT NULL,
   Type                 INT                  NOT NULL,
   DateFrom             datetime             NULL,
   DateTo               datetime             NULL,
   ExternalId           VARCHAR(20)          NULL,
   ExternalCentrId      VARCHAR(20)          NULL,
   LastUpdate           datetime
)
go

/*==============================================================*/
/* Table: CodeDefinitionElement                                 */
/*==============================================================*/
create table CodeDefinitionElement (
   CodeDefId            INTEGER              NOT NULL,
   Ordinal              INT                  NOT NULL,
   Type                 SMALLINT             NOT NULL,
   CodeLength           SMALLINT             NOT NULL,
   TextValue            text(40)             NULL,
   constraint PK_CodeDefinitionElement PRIMARY KEY (CodeDefId, Ordinal)
)
go

/* ============================================================ */
/*   Table: ProdOpak - na potrzeby SUP                          */
/* ============================================================ */
create table ProdOpak
(
   ProductId            integer              not null,
   Znaczenie            smallint             not null,
   Kolejnosc            smallint             not null,
   PackageId            integer              not null,
   Opcja1               smallint             null,
   Opcja2               smallint             null,
   Opcja3               smallint             null,
   Przelicznik          decimal(15,4)        null,
   MinIlosc             decimal(15,4)        null,
   MaxIlosc             decimal(15,4)        null,
   constraint PK_ProdOpak  primary key  (ProductId, Znaczenie, Kolejnosc)
)
go

/* ============================================================ */
/*   Table: FrakcjaDRS - na potrzeby DRS                        */
/* ============================================================ */
create table FrakcjaDRS (
   FrakId               integer              PRIMARY KEY AUTOINCREMENT,
   Nazwa                text(255)            not null,
   Kod                  text(40)             not null,
   Grupa                text(120)            not null,
   Aktywny              smallint             not null,
   CentrFrakId          integer              null,
   CenaZak              decimal(15,4)        null,
   CenaDet              decimal(15,4)        null,
   ExtOperator          text(40)             null,
   ExtId                text(40)             null,
   ExtAktywny           smallint             null,
   ExternalId           text(20)             null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null
)
go

/* ============================================================ */
/*   Table: PosOpakDRS - na potrzeby DRS                        */
/* ============================================================ */
create table PosOpakDRS
(
   PosOpakId            integer              PRIMARY KEY AUTOINCREMENT,
   Nazwa                text(255)            not null,
   EAN                  text(120)            not null,
   Aktywny              smallint             not null,
   CenaKaucji           decimal(15,4)        null,
   ProductId            integer              null,
   ExtOperator          text(40)             null,
   ExtOpakId            text(120)            null,
   ExtFrakId            text(40)             null,
   ExternalId           text(20)             null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null
)
go

/* ============================================================ */
/*   Table: PosDaneOpakDRS - na potrzeby DRS                    */
/* ============================================================ */
create table PosDaneOpakDRS
(
   PosOpakId            integer              not null,
   Znaczenie            smallint             not null,
   Kolejnosc            smallint             not null,
   Wartosc              text(255)            not null,
   constraint PK_PosDaneOpakDRS  primary key  (PosOpakId, Znaczenie, Kolejnosc)
)
go

/* ============================================================ */
/*   Table: PosKuponDRS - na potrzeby DRS                       */
/* ============================================================ */
create table PosKuponDRS
(
   PosKuponId           integer              PRIMARY KEY AUTOINCREMENT,
   ExtOperator          text(40)             null,
   KodKuponu            text(120)            not null,
   Status               smallint             not null,
   PrzydzialKiedy       datetime             null,
   PrzydzialPosId       integer              null,
   EmisjaKiedy          datetime             null,
   EmisjaPosId          integer              null,
   SpalenieKiedy        datetime             null,
   SpaleniePosId        integer              null,
   Creation             datetime             not null,
   LastUpdate           datetime             not null
)
go


/*==============================================================*/
/* Numer wersji bazy                                            */
/*==============================================================*/
insert into StructureVersion (Version, status) values (85, 0)
go

